set -euo pipefail

WARN=10
CRIT=5

# --- Helpers ---
log() { logger -t battery-watch "$*"; }

# Trouver une batterie (BAT0/BAT1)
BAT_DIR="$(find /sys/class/power_supply -maxdepth 1 -type l -name 'BAT*' | head -n1 || true)"
[[ -n "${BAT_DIR:-}" ]] || exit 0

CAP="$(<"$BAT_DIR/capacity")"
STATUS="$(<"$BAT_DIR/status")"   # Discharging/Charging/Full

# On agit seulement si ça se décharge
[[ "$STATUS" == "Discharging" ]] || exit 0

# --- Trouver un user "actif" (pour notify-send) ---
ACTIVE_USER=""
ACTIVE_UID=""
while read -r sid; do
  [[ -n "$sid" ]] || continue
  u="$(loginctl show-session "$sid" -p Name --value 2>/dev/null || true)"
  uid="$(id -u "$u" 2>/dev/null || true)"
  # bus DBus user (nécessaire pour notify-send)
  if [[ -n "$u" && -n "$uid" && -S "/run/user/$uid/bus" ]]; then
    ACTIVE_USER="$u"
    ACTIVE_UID="$uid"
    break
  fi
done < <(loginctl list-sessions --no-legend 2>/dev/null | awk '{print $1}')

# State persistant côté system (anti-spam)
STATE_BASE="/var/lib/battery-watch"
mkdir -p "$STATE_BASE"
STATE_DIR="$STATE_BASE/${ACTIVE_UID:-nouser}"
mkdir -p "$STATE_DIR"

notify_user() {
  local urgency="$1" title="$2" body="$3"
  [[ -n "$ACTIVE_USER" && -n "$ACTIVE_UID" ]] || return 0
  DBUS_SESSION_BUS_ADDRESS="unix:path=/run/user/$ACTIVE_UID/bus" \
    sudo -u "$ACTIVE_USER" /usr/bin/notify-send -u "$urgency" "$title" "$body" || true
}

if (( CAP <= WARN && CAP > CRIT )); then
  if [[ ! -f "$STATE_DIR/warned" ]]; then
    log "Battery low: ${CAP}% (WARN=${WARN}%)"
    notify_user critical "Batterie faible" "${CAP}% — branche-toi (alerte à ${WARN}%)."
    touch "$STATE_DIR/warned"
  fi
fi

if (( CAP <= CRIT )); then
  log "Battery critical: ${CAP}% (CRIT=${CRIT}%) -> poweroff"
  notify_user critical "Batterie critique" "${CAP}% — extinction automatique (${CRIT}%)."
  sleep 2
  systemctl poweroff
fi

# Reset anti-spam si on remonte au-dessus de WARN
if (( CAP > WARN )); then
  rm -f "$STATE_DIR/warned" 2>/dev/null || true
fi
