#!/usr/bin/env bash
set -euo pipefail

# =========================
# Helpers
# =========================
say()  { printf "\n\033[1;32m==>\033[0m %s\n" "$*"; }
warn() { printf "\n\033[1;33m[WARN]\033[0m %s\n" "$*"; }
die()  { printf "\n\033[1;31m[ERR]\033[0m %s\n" "$*" >&2; exit 1; }

need_cmd()  { command -v "$1" >/dev/null 2>&1 || die "Commande manquante: $1"; }
need_file() { [[ -f "$1" ]] || die "Fichier introuvable: $1"; }
need_dir()  { [[ -d "$1" ]] || die "Dossier introuvable: $1"; }

# =========================
# Paths
# =========================
SCRIPT_DIR="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" && pwd)"
SRC_ROOT="$(cd -- "${SCRIPT_DIR}/.." && pwd)"   # ~/savArch

CORE_LIST="${SCRIPT_DIR}/pkglist-core.txt"
APPS_LIST="${SCRIPT_DIR}/pkglist-apps.txt"

need_cmd sudo
need_cmd pacman
need_cmd rsync
need_cmd mkinitcpio

need_file "$CORE_LIST"
need_file "$APPS_LIST"
need_file "${SRC_ROOT}/.bashrc"
need_dir  "${SRC_ROOT}/.config"
need_dir  "${SRC_ROOT}/.local"

say "Source root : $SRC_ROOT"
say "Deploy dir  : $SCRIPT_DIR"
say "User        : $USER"
say "Home        : $HOME"

# =========================
# 1) Multilib
# =========================
say "Activation de [multilib]"
sudo sed -i \
  -e 's/^[[:space:]]*#[[:space:]]*\[multilib\]/[multilib]/' \
  -e 's|^[[:space:]]*#[[:space:]]*Include = /etc/pacman.d/mirrorlist|Include = /etc/pacman.d/mirrorlist|' \
  /etc/pacman.conf

# =========================
# 2) Paquets
# =========================
say "pacman sync + upgrade"
sudo pacman -Syyu --noconfirm

say "Installation CORE"
sudo pacman -S --needed --noconfirm - < "$CORE_LIST"

say "Installation APPS"
sudo pacman -S --needed --noconfirm - < "$APPS_LIST"

# =========================
# 3) Services essentiels
# =========================
say "Enable NetworkManager"
sudo systemctl enable --now NetworkManager.service || warn "NetworkManager non démarré"

say "Installation + activation seatd"
sudo pacman -S --needed --noconfirm seatd
sudo systemctl enable --now seatd.service || warn "seatd non démarré"

say "Ajout utilisateur aux groupes seat + video"
getent group seat  >/dev/null && sudo usermod -aG seat  "$USER" || true
getent group video >/dev/null && sudo usermod -aG video "$USER" || true

# =========================
# 4) NVIDIA
# =========================
say "Configuration NVIDIA DRM modeset"
echo "options nvidia-drm modeset=1" | sudo tee /etc/modprobe.d/nvidia.conf >/dev/null

say "Chargement modules NVIDIA (best effort)"
sudo modprobe nvidia         2>/dev/null || true
sudo modprobe nvidia_modeset 2>/dev/null || true
sudo modprobe nvidia_uvm     2>/dev/null || true
sudo modprobe nvidia_drm     2>/dev/null || true

say "mkinitcpio -P"
sudo mkinitcpio -P || warn "mkinitcpio a retourné une erreur"

# =========================
# 5) Dotfiles (COPY ONLY)
# =========================
say "Déploiement .bashrc"
cp -a "${SRC_ROOT}/.bashrc" "$HOME/.bashrc"

say "Déploiement ~/.config"
mkdir -p "$HOME/.config"
rsync -aAXH "${SRC_ROOT}/.config/" "$HOME/.config/"

say "Déploiement ~/.local"
mkdir -p "$HOME/.local"
rsync -aAXH "${SRC_ROOT}/.local/" "$HOME/.local/"

# IMPORTANT: éviter les soucis root:root si jamais une étape précédente a écrit en root
say "Fix ownership HOME (config/local) après rsync"
sudo chown -R "$USER:$USER" "$HOME/.config" "$HOME/.local" 2>/dev/null || true

# =========================
# 6) Firefox FIX (IMPORTANT)
# =========================
say "Initialisation Firefox (~/.mozilla/firefox)"
mkdir -p "$HOME/.mozilla/firefox"
chmod 700 "$HOME/.mozilla" 2>/dev/null || true
chmod 700 "$HOME/.mozilla/firefox" 2>/dev/null || true
sudo chown -R "$USER:$USER" "$HOME/.mozilla" 2>/dev/null || true

# =========================
# 6bis) AUR: installer yay (sans sudo) + asusctl via yay
# =========================
say "AUR: installation de yay (build en user)"
sudo pacman -S --needed --noconfirm git base-devel

if ! command -v yay >/dev/null 2>&1; then
  tmpdir="$(mktemp -d)"
  say "Clonage yay dans $tmpdir"
  git clone https://aur.archlinux.org/yay.git "$tmpdir/yay"
  (
    cd "$tmpdir/yay"
    # build/install en USER (pas de sudo ici)
    makepkg -si --noconfirm
  )
  rm -rf "$tmpdir"
else
  say "yay déjà installé"
fi

say "AUR: installation asusctl + supergfxctl via yay (pas de sudo sur yay)"
if command -v yay >/dev/null 2>&1; then
  # --needed: évite réinstall
  # --noconfirm: pas de prompts
  # --removemake: nettoie les deps de build
  yay -S --needed --noconfirm --removemake asusctl supergfxctl || warn "Installation asusctl/supergfxctl via AUR a échoué"
else
  warn "yay introuvable après installation; skip asusctl/supergfxctl"
fi

# Services ASUS (si install OK)
if systemctl list-unit-files | grep -q '^asusctl\.service'; then
  say "Enable/start asusctl"
  sudo systemctl enable --now asusctl.service || warn "asusctl service non démarré"
fi
if systemctl list-unit-files | grep -q '^supergfxd\.service'; then
  say "Enable/start supergfxd"
  sudo systemctl enable --now supergfxd.service || warn "supergfxd service non démarré"
fi

# =========================
# 7) Copie du deploy
# =========================
say "Copie du dossier deploy vers ~/deploy"
mkdir -p "$HOME/deploy"
rsync -aAXH "${SCRIPT_DIR}/" "$HOME/deploy/"

# =========================
# 8) Ownership final
# =========================
say "Fix ownership HOME (final)"
sudo chown -R "$USER:$USER" \
  "$HOME/.bashrc" \
  "$HOME/.config" \
  "$HOME/.local" \
  "$HOME/.mozilla" \
  "$HOME/deploy" 2>/dev/null || true

say "Déploiement terminé ✅"
echo
echo "⚠️ Reconnexion ou reboot requis pour les groupes (seat/video)"
echo "👉 Reboot conseillé : sudo reboot"
