#!/usr/bin/env bash
set -euo pipefail

##### Helpers #####

say()  { printf "\n\033[1;32m==>\033[0m %s\n" "$*"; }
warn() { printf "\n\033[1;33m[WARN]\033[0m %s\n" "$*"; }
die()  { printf "\n\033[1;31m[ERR]\033[0m %s\n" "$*" >&2; exit 1; }

need_cmd()  { command -v "$1" >/dev/null 2>&1 || die "Commande manquante: $1"; }
need_file() { [[ -f "$1" ]] || die "Fichier introuvable: $1"; }
need_dir()  { [[ -d "$1" ]] || die "Dossier introuvable: $1"; }
[[ $EUID -ne 0 ]] || die "Ne pas lancer en root"

##### Paths #####

SCRIPT_DIR="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" && pwd)"
SRC_ROOT="$(cd -- "${SCRIPT_DIR}/.." && pwd)"   # ~/savArch

CORE_LIST="${SCRIPT_DIR}/pkglist-core.txt"
APPS_LIST="${SCRIPT_DIR}/pkglist-apps.txt"
YAY_LIST="${SCRIPT_DIR}/pkglist-yay.txt"

need_cmd sudo
need_cmd pacman
need_cmd rsync
need_cmd mkinitcpio

need_file "$CORE_LIST"
need_file "$APPS_LIST"
need_file "${SRC_ROOT}/.bashrc"
need_file "${SRC_ROOT}/.bash_profile"
need_dir  "${SRC_ROOT}/.config"
need_dir  "${SRC_ROOT}/.local"

say "Source root : $SRC_ROOT"
say "Deploy dir  : $SCRIPT_DIR"
say "User        : $USER"
say "Home        : $HOME"

##### Multilib #####

say "Activation de [multilib] (append only)"

# Si un [multilib] non commenté existe déjà, on ne fait rien
if sudo grep -qE '^\[multilib\]\s*$' /etc/pacman.conf; then
  say "[multilib] déjà actif, rien à faire"
else
  # Sinon, on ajoute à la fin
  sudo tee -a /etc/pacman.conf >/dev/null <<'EOF'

[multilib]
Include = /etc/pacman.d/mirrorlist
EOF
  say "[multilib] ajouté à la fin"
fi

##### Paquets #####

say "pacman sync + upgrade"
sudo pacman -Syyu --noconfirm

say "Installation CORE"
sudo pacman -S --needed --noconfirm - < "$CORE_LIST"

say "Installation APPS"
sudo pacman -S --needed --noconfirm - < "$APPS_LIST"

##### Services essentiels #####

say "Enable NetworkManager"
sudo systemctl enable --now NetworkManager.service || warn "NetworkManager non démarré"

say "Enable seatd"
sudo systemctl enable --now seatd.service || warn "seatd non démarré"

say "Ajout utilisateur aux groupes seat + video"
getent group seat  >/dev/null && sudo usermod -aG seat  "$USER" || true
getent group video >/dev/null && sudo usermod -aG video "$USER" || true

say "Enable Bluetooth (BlueZ)"
sudo systemctl enable --now bluetooth.service || warn "Bluetooth non démarré"

##### TTY font (vconsole) #####

say "Configuration TTY font (vconsole)"
sudo tee /etc/vconsole.conf >/dev/null <<'EOF'
KEYMAP=be-latin1
FONT=ter-132n
EOF

##### NVIDIA #####

say "Configuration NVIDIA DRM modeset"
echo "options nvidia-drm modeset=1" | sudo tee /etc/modprobe.d/nvidia.conf >/dev/null

say "Chargement modules NVIDIA (best effort)"
sudo modprobe nvidia         2>/dev/null || true
sudo modprobe nvidia_modeset 2>/dev/null || true
sudo modprobe nvidia_uvm     2>/dev/null || true
sudo modprobe nvidia_drm     2>/dev/null || true

say "mkinitcpio -P"
sudo mkinitcpio -P || warn "mkinitcpio a retourné une erreur"

##### Dotfiles (COPY ONLY) #####

say "Déploiement .bashrc"
cp -a "${SRC_ROOT}/.bashrc" "$HOME/.bashrc"

say "Déploiement .bash_profile"
cp -a "${SRC_ROOT}/.bash_profile" "$HOME/.bash_profile"

say "Déploiement ~/.config"
mkdir -p "$HOME/.config"
rsync -aAXH "${SRC_ROOT}/.config/" "$HOME/.config/"

say "Déploiement ~/.local"
mkdir -p "$HOME/.local"
rsync -aAXH "${SRC_ROOT}/.local/" "$HOME/.local/"

##### Battery-watch (SYSTEMD) #####

say "Installation battery-watch (systemd system)"

sudo install -Dm0755 \
  "${SRC_ROOT}/assets/battery-watch/battery-watch.sh" \
  /usr/local/bin/battery-watch.sh

sudo install -Dm0644 \
  "${SRC_ROOT}/assets/battery-watch/battery-watch.service" \
  /etc/systemd/system/battery-watch.service

sudo install -Dm0644 \
  "${SRC_ROOT}/assets/battery-watch/battery-watch.timer" \
  /etc/systemd/system/battery-watch.timer

sudo systemctl daemon-reload
sudo systemctl enable --now battery-watch.timer

##### AUR: installer yay #####

say "AUR: installation de yay"

if ! command -v yay >/dev/null 2>&1; then
  tmpdir="$(mktemp -d)"
  git clone https://aur.archlinux.org/yay.git "$tmpdir/yay"
  (
    cd "$tmpdir/yay"
    makepkg -si --noconfirm
  )
  rm -rf "$tmpdir"
else
  say "yay déjà installé"
fi

if command -v yay >/dev/null 2>&1 && [[ -f "$YAY_LIST" ]]; then
  say "AUR: installation des paquets depuis pkglist-yay.txt"
  yay -S --needed --noconfirm --removemake - < "$YAY_LIST" \
    || warn "Installation AUR partiellement échouée"
else
  warn "yay ou pkglist-yay.txt manquant, skip AUR"
fi

##### Services ASUS #####

if systemctl list-unit-files | grep -q '^asusctl\.service'; then
  say "Enable/start asusctl"
  sudo systemctl enable --now asusctl.service || warn "asusctl non démarré"
fi

if systemctl list-unit-files | grep -q '^supergfxd\.service'; then
  say "Enable/start supergfxd"
  sudo systemctl enable --now supergfxd.service || warn "supergfxd non démarré"
fi

##### Ownership final #####

say "Fix ownership HOME (final)"
sudo chown -R "$USER:$USER" \
  "$HOME/.bashrc" \
  "$HOME/.bash_profile" \
  "$HOME/.config" \
  "$HOME/.local" 2>/dev/null || true

say "+X fichier sh .local/bin/"
find "$HOME/.local/bin" -maxdepth 1 -type f -exec chmod 0755 {} \;

say "Déploiement terminé "
echo "👉 Reboot conseillé : sudo reboot"
echo "✅ Battery-watch (system) actif : systemctl status battery-watch.timer"
