#!/usr/bin/env bash
set -euo pipefail

# =========================
# Helpers
# =========================
say()  { printf "\n\033[1;32m==>\033[0m %s\n" "$*"; }
warn() { printf "\n\033[1;33m[WARN]\033[0m %s\n" "$*"; }
die()  { printf "\n\033[1;31m[ERR]\033[0m %s\n" "$*" >&2; exit 1; }

need_cmd()  { command -v "$1" >/dev/null 2>&1 || die "Commande manquante: $1"; }
need_file() { [[ -f "$1" ]] || die "Fichier introuvable: $1"; }
need_dir()  { [[ -d "$1" ]] || die "Dossier introuvable: $1"; }

# =========================
# Paths
# =========================
SCRIPT_DIR="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" && pwd)"
SRC_ROOT="$(cd -- "${SCRIPT_DIR}/.." && pwd)"   # ~/savArch

CORE_LIST="${SCRIPT_DIR}/pkglist-core.txt"
APPS_LIST="${SCRIPT_DIR}/pkglist-apps.txt"

need_cmd sudo
need_cmd pacman
need_cmd rsync
need_cmd mkinitcpio

need_file "$CORE_LIST"
need_file "$APPS_LIST"
need_file "${SRC_ROOT}/.bashrc"
need_file "${SRC_ROOT}/.bash_profile"
need_dir  "${SRC_ROOT}/.config"
need_dir  "${SRC_ROOT}/.local"

say "Source root : $SRC_ROOT"
say "Deploy dir  : $SCRIPT_DIR"
say "User        : $USER"
say "Home        : $HOME"

# =========================
# 1) Multilib
# =========================

say "Activation de [multilib] (append only)"

# Si un [multilib] actif existe déjà, on ne fait rien
if sudo grep -qE '^\[multilib\]\s*$' /etc/pacman.conf; then
  say "[multilib] déjà actif, rien à faire"
else
  # Sinon, on ajoute à la fin
  sudo tee -a /etc/pacman.conf >/dev/null <<'EOF'

[multilib]
Include = /etc/pacman.d/mirrorlist
EOF
  say "[multilib] ajouté à la fin"
fi

# =========================
# 2) Paquets
# =========================
say "pacman sync + upgrade"
sudo pacman -Syyu --noconfirm

say "Installation CORE"
sudo pacman -S --needed --noconfirm - < "$CORE_LIST"

say "Installation APPS"
sudo pacman -S --needed --noconfirm - < "$APPS_LIST"

# Notifs (dunst + notify-send)
say "Installation notifications (dunst + libnotify)"
sudo pacman -S --needed --noconfirm dunst libnotify || warn "Impossible d'installer dunst/libnotify"

# =========================
# 3) Services essentiels
# =========================
say "Enable NetworkManager"
sudo systemctl enable --now NetworkManager.service || warn "NetworkManager non démarré"

say "Installation + activation seatd"
sudo pacman -S --needed --noconfirm seatd
sudo systemctl enable --now seatd.service || warn "seatd non démarré"

say "Ajout utilisateur aux groupes seat + video"
getent group seat  >/dev/null && sudo usermod -aG seat  "$USER" || true
getent group video >/dev/null && sudo usermod -aG video "$USER" || true

# =========================
# TTY font (vconsole)
# =========================
say "Configuration TTY font (vconsole)"
sudo tee /etc/vconsole.conf >/dev/null <<'EOF'
KEYMAP=be-latin1
FONT=ter-132n
EOF

# =========================
# 4) NVIDIA
# =========================
say "Configuration NVIDIA DRM modeset"
echo "options nvidia-drm modeset=1" | sudo tee /etc/modprobe.d/nvidia.conf >/dev/null

say "Chargement modules NVIDIA (best effort)"
sudo modprobe nvidia         2>/dev/null || true
sudo modprobe nvidia_modeset 2>/dev/null || true
sudo modprobe nvidia_uvm     2>/dev/null || true
sudo modprobe nvidia_drm     2>/dev/null || true

say "mkinitcpio -P"
sudo mkinitcpio -P || warn "mkinitcpio a retourné une erreur"

# =========================
# 5) Dotfiles (COPY ONLY)
# =========================
say "Déploiement .bashrc"
cp -a "${SRC_ROOT}/.bashrc" "$HOME/.bashrc"

say "Déploiement .bash_profile"
cp -a "${SRC_ROOT}/.bash_profile" "$HOME/.bash_profile"

say "Déploiement ~/.config"
mkdir -p "$HOME/.config"
rsync -aAXH "${SRC_ROOT}/.config/" "$HOME/.config/"

say "Déploiement ~/.local"
mkdir -p "$HOME/.local"
rsync -aAXH "${SRC_ROOT}/.local/" "$HOME/.local/"

# IMPORTANT: éviter les soucis root:root si jamais une étape précédente a écrit en root
say "Fix ownership HOME (config/local) après rsync"
sudo chown -R "$USER:$USER" "$HOME/.config" "$HOME/.local" 2>/dev/null || true

# =========================
# Battery-watch (SYSTEMD SYSTEM) + cleanup USER
# =========================
say "Cleanup ancien battery-watch (systemd --user) (best effort)"
systemctl --user disable --now battery-watch.timer 2>/dev/null || true
systemctl --user stop battery-watch.service 2>/dev/null || true
rm -f "$HOME/.config/systemd/user/battery-watch.service" \
      "$HOME/.config/systemd/user/battery-watch.timer" \
      "$HOME/.config/systemd/user/timers.target.wants/battery-watch.timer" 2>/dev/null || true
systemctl --user daemon-reload 2>/dev/null || true
systemctl --user reset-failed 2>/dev/null || true

say "Installation battery-watch (systemd system)"

# Script system-safe (log + notif si session user DBus présente + poweroff)
sudo tee /usr/local/bin/battery-watch.sh >/dev/null <<'EOF'
#!/usr/bin/env bash
set -euo pipefail

WARN=10
CRIT=5

log() { logger -t battery-watch "$*"; }

BAT_DIR="$(find /sys/class/power_supply -maxdepth 1 -type l -name 'BAT*' | head -n1 || true)"
[[ -n "${BAT_DIR:-}" ]] || exit 0

CAP="$(<"$BAT_DIR/capacity")"
STATUS="$(<"$BAT_DIR/status")"

[[ "$STATUS" == "Discharging" ]] || exit 0

ACTIVE_USER=""
ACTIVE_UID=""

while read -r sid; do
  [[ -n "$sid" ]] || continue
  u="$(loginctl show-session "$sid" -p Name --value 2>/dev/null || true)"
  uid="$(id -u "$u" 2>/dev/null || true)"
  if [[ -n "$u" && -n "$uid" && -S "/run/user/$uid/bus" ]]; then
    ACTIVE_USER="$u"
    ACTIVE_UID="$uid"
    break
  fi
done < <(loginctl list-sessions --no-legend 2>/dev/null | awk '{print $1}')

STATE_BASE="/var/lib/battery-watch"
mkdir -p "$STATE_BASE"
STATE_DIR="$STATE_BASE/${ACTIVE_UID:-nouser}"
mkdir -p "$STATE_DIR"

notify_user() {
  local urgency="$1" title="$2" body="$3"
  [[ -n "$ACTIVE_USER" && -n "$ACTIVE_UID" ]] || return 0
  DBUS_SESSION_BUS_ADDRESS="unix:path=/run/user/$ACTIVE_UID/bus" \
    sudo -u "$ACTIVE_USER" /usr/bin/notify-send -u "$urgency" "$title" "$body" || true
}

if (( CAP <= WARN && CAP > CRIT )); then
  if [[ ! -f "$STATE_DIR/warned" ]]; then
    log "Battery low: ${CAP}% (WARN=${WARN}%)"
    notify_user critical "Batterie faible" "${CAP}% — branche-toi (alerte à ${WARN}%)."
    touch "$STATE_DIR/warned"
  fi
fi

if (( CAP <= CRIT )); then
  log "Battery critical: ${CAP}% (CRIT=${CRIT}%) -> poweroff"
  notify_user critical "Batterie critique" "${CAP}% — extinction automatique (${CRIT}%)."
  sleep 2
  systemctl poweroff
fi

if (( CAP > WARN )); then
  rm -f "$STATE_DIR/warned" 2>/dev/null || true
fi
EOF

sudo chmod 0755 /usr/local/bin/battery-watch.sh
sudo mkdir -p /var/lib/battery-watch

sudo tee /etc/systemd/system/battery-watch.service >/dev/null <<'EOF'
[Unit]
Description=Battery watch (notify at 10%, shutdown at 5%)

[Service]
Type=oneshot
ExecStart=/usr/local/bin/battery-watch.sh
EOF

sudo tee /etc/systemd/system/battery-watch.timer >/dev/null <<'EOF'
[Unit]
Description=Run battery watch every 2 minutes

[Timer]
OnBootSec=2min
OnUnitActiveSec=2min
AccuracySec=30s

[Install]
WantedBy=timers.target
EOF

sudo systemctl daemon-reload
sudo systemctl enable --now battery-watch.timer

# =========================
# 6) Firefox FIX (IMPORTANT)
# =========================
say "Initialisation Firefox (~/.mozilla/firefox)"
mkdir -p "$HOME/.mozilla/firefox"
chmod 700 "$HOME/.mozilla" 2>/dev/null || true
chmod 700 "$HOME/.mozilla/firefox" 2>/dev/null || true
sudo chown -R "$USER:$USER" "$HOME/.mozilla" 2>/dev/null || true

# =========================
# 6bis) AUR: installer yay (sans sudo) + asusctl via yay
# =========================
say "AUR: installation de yay (build en user)"
sudo pacman -S --needed --noconfirm git base-devel

if ! command -v yay >/dev/null 2>&1; then
  tmpdir="$(mktemp -d)"
  say "Clonage yay dans $tmpdir"
  git clone https://aur.archlinux.org/yay.git "$tmpdir/yay"
  (
    cd "$tmpdir/yay"
    makepkg -si --noconfirm
  )
  rm -rf "$tmpdir"
else
  say "yay déjà installé"
fi

say "AUR: installation asusctl + supergfxctl via yay (pas de sudo sur yay)"
if command -v yay >/dev/null 2>&1; then
  yay -S --needed --noconfirm --removemake asusctl supergfxctl || warn "Installation asusctl/supergfxctl via AUR a échoué"
else
  warn "yay introuvable après installation; skip asusctl/supergfxctl"
fi

# Services ASUS (si install OK)
if systemctl list-unit-files | grep -q '^asusctl\.service'; then
  say "Enable/start asusctl"
  sudo systemctl enable --now asusctl.service || warn "asusctl service non démarré"
fi
if systemctl list-unit-files | grep -q '^supergfxd\.service'; then
  say "Enable/start supergfxd"
  sudo systemctl enable --now supergfxd.service || warn "supergfxd service non démarré"
fi

# =========================
# 7) Copie du deploy
# =========================
say "Copie du dossier deploy vers ~/deploy"
mkdir -p "$HOME/deploy"
rsync -aAXH "${SCRIPT_DIR}/" "$HOME/deploy/"

# =========================
# 8) Ownership final
# =========================
say "Fix ownership HOME (final)"
sudo chown -R "$USER:$USER" \
  "$HOME/.bashrc" \
  "$HOME/.config" \
  "$HOME/.local" \
  "$HOME/.mozilla" \
  "$HOME/deploy" 2>/dev/null || true

say "Déploiement terminé ✅"
echo
echo "⚠️ Reconnexion ou reboot requis pour les groupes (seat/video)"
echo "👉 Reboot conseillé : sudo reboot"
echo "✅ Battery-watch (system) actif : systemctl status battery-watch.timer"
