#!/usr/bin/env bash
set -euo pipefail

# ==============================================================================
# Deploy Arch user environment + packages
# Source layout expected:
#   ~/savArch/
#     .bashrc
#     .config/
#     .local/
#     deploy/
#       pkglist-core.txt
#       pkglist-apps.txt
#       deploy.bash   (this script)
# ==============================================================================

say() { printf "\n\033[1;32m==>\033[0m %s\n" "$*"; }
warn() { printf "\n\033[1;33m[WARN]\033[0m %s\n" "$*"; }
die() { printf "\n\033[1;31m[ERR]\033[0m %s\n" "$*" >&2; exit 1; }

need_cmd() { command -v "$1" >/dev/null 2>&1 || die "Commande manquante: $1"; }
need_file() { [[ -f "$1" ]] || die "Fichier introuvable: $1"; }
need_dir() { [[ -d "$1" ]] || die "Dossier introuvable: $1"; }

backup_if_exists() {
  local p="$1"
  if [[ -e "$p" && ! -L "$p" ]]; then
    local ts
    ts="$(date +%Y%m%d-%H%M%S)"
    local bak="${p}.bak.${ts}"
    say "Backup: $p -> $bak"
    mv "$p" "$bak"
  fi
}

rsync_copy_dir() {
  local src="$1"
  local dst="$2"
  need_dir "$src"
  mkdir -p "$dst"
  # Copy contents, preserve perms, xattrs, ACLs, delete removed files
  rsync -aAXH --delete "${src%/}/" "${dst%/}/"
}

# --- Prechecks ---
need_cmd sudo
need_cmd pacman
need_cmd rsync
need_cmd mkinitcpio

SCRIPT_DIR="$(cd -- "$(dirname -- "${BASH_SOURCE[0]}")" && pwd)"
DEPLOY_DIR="$SCRIPT_DIR"
SAVARCH_DIR="$(cd -- "${DEPLOY_DIR}/.." && pwd)"

CORE_LIST="${DEPLOY_DIR}/pkglist-core.txt"
APPS_LIST="${DEPLOY_DIR}/pkglist-apps.txt"

need_file "$CORE_LIST"
need_file "$APPS_LIST"
need_file "${SAVARCH_DIR}/.bashrc"
need_dir  "${SAVARCH_DIR}/.config"
need_dir  "${SAVARCH_DIR}/.local"

say "Source: $SAVARCH_DIR"
say "Deploy: $DEPLOY_DIR"
say "User:   $USER"
say "Home:   $HOME"

# --- 1) Sync system & install packages ---
say "Sync + upgrade (pacman -Syu)"
sudo pacman -Syu --noconfirm

say "Install CORE packages"
sudo pacman -S --needed - < "$CORE_LIST"

say "Install APPS packages"
sudo pacman -S --needed - < "$APPS_LIST"

# --- 2) Enable services that should exist after reset ---
if systemctl list-unit-files | grep -q '^NetworkManager\.service'; then
  say "Enable NetworkManager"
  sudo systemctl enable NetworkManager --now || warn "NetworkManager enable failed (ok si déjà actif)"
fi

# --- 3) NVIDIA modeset + initramfs (safe even if AMD-only; file is harmless) ---
say "Configure NVIDIA DRM modeset=1"
echo "options nvidia-drm modeset=1" | sudo tee /etc/modprobe.d/nvidia.conf >/dev/null

say "Rebuild initramfs (mkinitcpio -P)"
sudo mkinitcpio -P || warn "mkinitcpio a retourné une erreur (souvent si module nvidia absent). Vérifie nvidia-dkms + linux-headers."

# --- 4) Deploy dotfiles / configs to HOME (with backups) ---
say "Deploy ~/.bashrc"
backup_if_exists "$HOME/.bashrc"
install -m 0644 "${SAVARCH_DIR}/.bashrc" "$HOME/.bashrc"

say "Deploy ~/.config (rsync)"
backup_if_exists "$HOME/.config"
rsync_copy_dir "${SAVARCH_DIR}/.config" "$HOME/.config"

say "Deploy ~/.local (rsync)"
backup_if_exists "$HOME/.local"
rsync_copy_dir "${SAVARCH_DIR}/.local" "$HOME/.local"

# --- 5) Copy the deploy folder into ~/deploy (handy for re-run) ---
say "Copy deploy folder to ~/deploy"
backup_if_exists "$HOME/deploy"
mkdir -p "$HOME/deploy"
rsync -aAXH --delete "${DEPLOY_DIR%/}/" "${HOME%/}/deploy/"

# --- 6) Ownership fix (in case this was run after restoring as root) ---
say "Fix ownership in HOME (best effort)"
sudo chown -R "$USER:$USER" "$HOME/.config" "$HOME/.local" "$HOME/.bashrc" "$HOME/deploy" 2>/dev/null || true

say "DONE ✅"
echo
echo "Prochaines vérifs conseillées :"
echo "  - glxinfo -B"
echo "  - lsmod | grep nvidia"
echo "  - prism-launcher"
echo
echo "Reboot conseillé : sudo reboot"

