#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import io
import re
import logging
import asyncio
import json
import time
from dataclasses import dataclass
from typing import List, Dict, Optional, Union, Tuple

import requests
from bs4 import BeautifulSoup
from urllib.parse import quote, urljoin, urlparse, parse_qs, urlencode

from telegram import (
    Update,
    InlineKeyboardMarkup,
    InlineKeyboardButton,
)
from telegram.error import BadRequest
from telegram.ext import (
    ApplicationBuilder,
    CommandHandler,
    MessageHandler,
    CallbackQueryHandler,
    ContextTypes,
    ConversationHandler,
    filters,
)

# ======= LOGGING =======
logging.basicConfig(
    format="%(asctime)s | %(levelname)s | %(name)s | %(message)s",
    level=logging.INFO,
)
logger = logging.getLogger(__name__)

# ======= ETATS CONVERSATION =======
MODE, SEARCH, CHOOSE_RESULT = range(3)

# ======= CONFIG ENV =======
TELEGRAM_TOKEN = os.getenv("TELEGRAM_TOKEN", "").strip()
INDEX_BASE_URL = os.getenv("INDEX_BASE_URL", "https://zone-telechargement.irish").rstrip("/")

FLARESOLVERR_URL = os.getenv("FLARESOLVERR_URL", "http://flaresolverr:8191/v1")

DOWNLOAD_DIR_FILMS = os.getenv("DEST_FILMS", "/app/films")
DOWNLOAD_DIR_SERIES = os.getenv("DEST_SERIES", "/app/series")

os.makedirs(DOWNLOAD_DIR_FILMS, exist_ok=True)
os.makedirs(DOWNLOAD_DIR_SERIES, exist_ok=True)

ALLDEBRID_API_KEY = os.getenv("ALLDEBRID_API_KEY", "").strip()
ALLDEBRID_AGENT = os.getenv("ALLDEBRID_AGENT", "dhdflix").strip()

session = requests.Session()

# Torrent indexer config (YGG-like)
TORRENT_SITE_BASE_URL = os.getenv("TORRENT_SITE_BASE_URL", "").rstrip("/")
TORRENT_USERNAME = os.getenv("TORRENT_USERNAME", "").strip()
TORRENT_PASSWORD = os.getenv("TORRENT_PASSWORD", "").strip()

# Default categories for films and series in the torrent indexer
TORRENT_CATEGORY_FILM = int(os.getenv("TORRENT_CATEGORY_FILM", "2145"))
TORRENT_SUBCATEGORY_FILM = os.getenv("TORRENT_SUBCATEGORY_FILM", "2183")  # Film/Vidéo : Film
TORRENT_CATEGORY_SERIE = int(os.getenv("TORRENT_CATEGORY_SERIE", "2145"))
TORRENT_SUBCATEGORY_SERIE = os.getenv("TORRENT_SUBCATEGORY_SERIE", "2184")  # Film/Vidéo : Série TV
session.headers.update(
    {
        "User-Agent": (
            "Mozilla/5.0 (Windows NT 10.0; Win64; x64) "
            "AppleWebKit/537.36 (KHTML, like Gecko) "
            "Chrome/120.0 Safari/537.36"
        )
    }
)


# ======= DATACLASSES =======
@dataclass
class SearchResult:
    title: str
    quality: str
    url: str
    is_series: bool



@dataclass
class TorrentResult:
    title: str
    detail_url: str
    size: Optional[str] = None
    seeds: Optional[int] = None
    leechs: Optional[int] = None


# ======= TORRENT INTEGRATION (index + AllDebrid) =======
@dataclass
class Config:
    site_base_url: str
    flaresolverr_url: str
    site_username: str
    site_password: str
    default_category: int
    default_sub_category: Union[int, str]
    download_dir: str
    alldebrid_api_key: Optional[str] = None
    telegram_token: Optional[str] = None




class FlaresolverrClient:
    def __init__(self, base_url: str, max_timeout: int = 60000):
        self.base_url = base_url.rstrip("/")
        self.max_timeout = max_timeout
        self.session_id: Optional[str] = None
        self.cookies: Optional[requests.cookies.RequestsCookieJar] = None
        self.user_agent: Optional[str] = None

    def create_session(self) -> None:
        payload = {
            "cmd": "sessions.create",
            "maxTimeout": self.max_timeout
        }
        try:
            resp = requests.post(self.base_url, json=payload, timeout=60)
        except requests.exceptions.RequestException as e:
            raise RuntimeError(f"Impossible de joindre Flaresolverr à {self.base_url} : {e}")
        resp.raise_for_status()
        data = resp.json()
        if data.get("status") != "ok":
            raise RuntimeError(f"Erreur Flaresolverr sessions.create: {data}")
        self.session_id = data["session"]

    def destroy_session(self) -> None:
        if not self.session_id:
            return
        payload = {
            "cmd": "sessions.destroy",
            "session": self.session_id
        }
        try:
            requests.post(self.base_url, json=payload, timeout=30)
        except Exception:
            pass
        self.session_id = None

    def _request(self, method: str, url: str, **kwargs) -> str:
        if not self.session_id:
            self.create_session()

        payload = {
            "cmd": f"request.{method}",
            "url": url,
            "session": self.session_id,
            "maxTimeout": self.max_timeout,
        }

        headers = kwargs.get("headers")
        data = kwargs.get("data")

        if headers:
            payload["headers"] = headers
        if data is not None:
            payload["postData"] = urlencode(data)

        resp = requests.post(self.base_url, json=payload, timeout=self.max_timeout / 1000 + 10)
        resp.raise_for_status()
        data = resp.json()
        if data.get("status") != "ok":
            raise RuntimeError(f"Erreur Flaresolverr request.{method}: {data}")
        solution = data.get("solution", {})

        cookies_list = solution.get("cookies", [])
        if cookies_list:
            jar = requests.cookies.RequestsCookieJar()
            for c in cookies_list:
                name = c.get("name")
                value = c.get("value")
                if not name:
                    continue
                domain = c.get("domain")
                path = c.get("path", "/")
                jar.set(name, value, domain=domain, path=path)
            self.cookies = jar

        ua = solution.get("userAgent")
        if ua:
            self.user_agent = ua

        return solution.get("response", "")

    def get(self, url: str) -> str:
        return self._request("get", url)

    def post(self, url: str, data: dict, headers: Optional[dict] = None) -> str:
        headers = headers or {"Content-Type": "application/x-www-form-urlencoded"}
        return self._request("post", url, data=data, headers=headers)

    def download_file(self, url: str, dest_path: str) -> None:
        if self.user_agent:
            headers = {"User-Agent": self.user_agent}
        else:
            headers = {}

        cookies = self.cookies or {}

        with requests.get(url, headers=headers, cookies=cookies, stream=True, timeout=120) as r:
            r.raise_for_status()
            with open(dest_path, "wb") as f:
                for chunk in r.iter_content(chunk_size=8192):
                    if chunk:
                        f.write(chunk)


# =========================
# SITE CLIENT
# =========================



class SiteClient:
    def __init__(self, config: Config, fs: FlaresolverrClient):
        self.config = config
        self.fs = fs

    @property
    def base(self) -> str:
        return self.config.site_base_url.rstrip("/")

    def login(self) -> None:
        login_url = urljoin(self.base, "/auth/process_login")
        data = {
            "id": self.config.site_username,
            "pass": self.config.site_password,
        }
        html = self.fs.post(login_url, data=data)

        if "Mauvais" in html or "erreur" in html.lower():
            raise RuntimeError("Login échoué, vérifie les identifiants / le sélecteur.")
        print("[+] Connecté (vérification basique).")

    def _build_search_url(self, name: str, category: int, sub_category: Union[int, str]) -> str:
        params = {
            "name": name,
            "description": "",
            "file": "",
            "uploader": "",
            "category": category,
            "sub_category": sub_category if sub_category is not None else "all",
            "do": "search"
        }
        return self.base + "/engine/search?" + urlencode(params)

    def _parse_results_page(self, html: str) -> List[TorrentResult]:
        soup = BeautifulSoup(html, "html.parser")
        results: List[TorrentResult] = []

        container = soup.find("div", class_="table-responsive results")
        if not container:
            return results

        table = container.find("table")
        if not table:
            return results

        tbody = table.find("tbody")
        if not tbody:
            return results

        for row in tbody.find_all("tr"):
            cols = row.find_all("td")
            if len(cols) < 2:
                continue

            name_td = cols[1]
            link = name_td.find("a", id="torrent_name")
            if not link or not link.get("href"):
                continue

            title = link.get_text(strip=True)
            href = link["href"]
            if href.startswith("/"):
                detail_url = urljoin(self.base, href)
            else:
                detail_url = href

            size = None
            seeds = None
            leechs = None

            if len(cols) >= 6:
                size_txt = cols[5].get_text(strip=True)
                size = size_txt or None

            if len(cols) >= 8:
                seeds_txt = cols[7].get_text(strip=True).replace(" ", "")
                if seeds_txt.isdigit():
                    seeds = int(seeds_txt)

            if len(cols) >= 9:
                leech_txt = cols[8].get_text(strip=True).replace(" ", "")
                if leech_txt.isdigit():
                    leechs = int(leech_txt)

            results.append(
                TorrentResult(
                    title=title,
                    detail_url=detail_url,
                    size=size,
                    seeds=seeds,
                    leechs=leechs,
                )
            )

        return results

    def _get_pagination_links(self, html: str) -> List[str]:
        soup = BeautifulSoup(html, "html.parser")
        ul = soup.find("ul", class_="pagination")
        if not ul:
            return []

        page_entries: List[Tuple[int, str]] = []
        seen = set()

        for a in ul.find_all("a"):
            page_attr = a.get("data-ci-pagination-page")
            href = a.get("href")
            if not page_attr or not href:
                continue

            try:
                page_index = int(page_attr)
            except ValueError:
                continue

            if page_index <= 1:
                continue

            if href.startswith("/"):
                href = urljoin(self.base, href)

            key = (page_index, href)
            if key in seen:
                continue
            seen.add(key)
            page_entries.append((page_index, href))

        page_entries.sort(key=lambda x: x[0])
        return [href for _, href in page_entries]

    def search(self, name: str, category: Optional[int], sub_category: Optional[Union[int, str]]) -> List[TorrentResult]:
        cat = category if category is not None else self.config.default_category
        sub_cat: Union[int, str] = sub_category if sub_category is not None else self.config.default_sub_category

        first_url = self._build_search_url(name, cat, sub_cat)
        print(f"[+] Recherche : {first_url}")
        first_html = self.fs.get(first_url)

        results = self._parse_results_page(first_html)
        page_urls = self._get_pagination_links(first_html)
        total_pages = 1 + len(page_urls)
        print(f"[+] Page 1/{total_pages} : {len(results)} résultats")

        for idx, page_url in enumerate(page_urls, start=2):
            print(f"[+] Page {idx}/{total_pages} -> {page_url}")
            html = self.fs.get(page_url)
            page_results = self._parse_results_page(html)
            print(f"    -> {len(page_results)} résultats")
            results.extend(page_results)

        return results

    def get_download_url_from_detail(self, detail_url: str) -> Optional[str]:
        html = self.fs.get(detail_url)
        soup = BeautifulSoup(html, "html.parser")

        link = None
        for a in soup.find_all("a"):
            txt = a.get_text(strip=True).lower()
            if "télécharger le torrent" in txt:
                link = a
                break

        if not link or not link.get("href"):
            print("[-] Impossible de trouver le lien de téléchargement du torrent.")
            return None

        href = link["href"]
        if href.startswith("/"):
            href = urljoin(self.base, href)

        return href


# =========================
# ALLDEBRID CLIENT
# =========================



class AllDebridClient:
    """
    Client AllDebrid :
    - upload du .torrent
    - attente du statut "Ready"
    - récupération de tous les fichiers
    - téléchargement dans download_dir/<nom_torrent>/
    """

    def __init__(self, api_key: str):
        self.api_key = api_key
        self.base = "https://api.alldebrid.com"

    def _headers(self) -> dict:
        return {"Authorization": f"Bearer {self.api_key}"}

    def _request(self, method: str, path: str, **kwargs) -> dict:
        url = self.base + path
        headers = kwargs.pop("headers", {})
        headers.update(self._headers())

        resp = requests.request(method, url, headers=headers, timeout=60, **kwargs)
        resp.raise_for_status()
        try:
            payload = resp.json()
        except json.JSONDecodeError:
            raise RuntimeError(f"Réponse non JSON AllDebrid sur {path} : {resp.text[:500]}")
        status = payload.get("status")
        if status != "success":
            raise RuntimeError(f"Erreur AllDebrid sur {path} : {payload}")
        return payload.get("data", {})

    # ---------- Upload .torrent ----------

    def upload_torrent_file(self, torrent_path: str) -> Tuple[int, bool]:
        path = "/v4/magnet/upload/file"
        filename = os.path.basename(torrent_path)

        with open(torrent_path, "rb") as f:
            files = {
                "files[]": (filename, f, "application/x-bittorrent"),
            }
            data = self._request("POST", path, files=files)

        print(f"[AllDebrid][DEBUG] Réponse /magnet/upload/file : {json.dumps(data, indent=2)[:1000]}")

        files_info = data.get("files", [])
        if not files_info:
            raise RuntimeError(f"Aucun fichier retourné par /magnet/upload/file : {data}")

        for fi in files_info:
            if fi.get("error"):
                continue
            magnet_id = fi.get("id")
            ready = bool(fi.get("ready", False))
            if magnet_id is not None:
                return int(magnet_id), ready

        raise RuntimeError(f"Aucun magnet valide dans la réponse : {files_info}")

    # ---------- Poll statut ----------

    def wait_until_ready(self, magnet_id: int, poll_interval: int = 5, timeout: int = 1800) -> dict:
        path = "/v4.1/magnet/status"
        elapsed = 0

        while elapsed < timeout:
            data = self._request("POST", path, data={"id": magnet_id})
            print(f"[AllDebrid][DEBUG] Réponse brute /magnet/status : {json.dumps(data, indent=2)[:1000]}")

            magnets = data.get("magnets")

            # magnets peut être un dict (ton cas) ou une liste
            if isinstance(magnets, dict):
                magnet = magnets
            elif isinstance(magnets, list) and magnets:
                magnet = magnets[0]
            else:
                print("[AllDebrid] Aucun magnet dans la réponse, on réessaie…")
                magnet = None

            if magnet:
                status_code = magnet.get("statusCode")
                status_str = magnet.get("status", "Unknown")
                print(f"[AllDebrid] Statut magnet {magnet_id} : {status_str} (code {status_code})")

                if status_code == 4:
                    print("[AllDebrid] Magnet prêt.")
                    return magnet

                if status_code is not None and status_code >= 5:
                    raise RuntimeError(
                        f"Magnet en erreur côté AllDebrid (code {status_code}, statut {status_str})"
                    )

            time.sleep(poll_interval)
            elapsed += poll_interval

        raise TimeoutError(f"Le magnet {magnet_id} n'est pas prêt après {timeout} secondes.")

    # ---------- Récupération fichiers ----------

    def _flatten_files_tree(self, nodes, prefix: str = "") -> List[dict]:
        flat: List[dict] = []
        for node in nodes:
            if not isinstance(node, dict):
                continue
            name = node.get("n", "")
            path = f"{prefix}/{name}" if prefix else name

            if "e" in node:  # dossier
                flat.extend(self._flatten_files_tree(node.get("e", []), path))
            else:  # fichier
                link = node.get("l")
                size = int(node.get("s", 0) or 0)
                if link:
                    flat.append({"name": path, "size": size, "link": link})
        return flat

    def get_files_links(self, magnet_id: int) -> List[dict]:
        path = "/v4.1/magnet/files"
        data = self._request("POST", path, data={"id[]": [magnet_id]})

        print(f"[AllDebrid][DEBUG] Réponse brute /magnet/files : {json.dumps(data, indent=2)[:2000]}")

        magnets = data.get("magnets")

        # Peut être dict ou liste
        if isinstance(magnets, dict):
            magnet_obj = magnets
        elif isinstance(magnets, list) and magnets:
            magnet_obj = magnets[0]
        else:
            raise RuntimeError(f"Aucun magnet dans /magnet/files pour ID {magnet_id} : {data}")

        files_tree = magnet_obj.get("files", [])
        flat_files = self._flatten_files_tree(files_tree)

        print(f"[AllDebrid] {len(flat_files)} fichier(s) détecté(s) dans le magnet.")
        for f in flat_files:
            print(f"  - {f['name']} ({f['size']} octets)")

        return flat_files

    # ---------- Téléchargement ----------

    def download_direct(self, url: str, dest_path: str, progress_cb=None) -> None:
        """Téléchargement d'un fichier AllDebrid (torrent) avec callback de progression optionnel."""
        os.makedirs(os.path.dirname(dest_path) or ".", exist_ok=True)
        print(f"[AllDebrid] Téléchargement -> {dest_path}")
        with requests.get(url, stream=True, timeout=600) as r:
            r.raise_for_status()
            total = int(r.headers.get("Content-Length") or 0)
            downloaded = 0
            last_pct = -1
            with open(dest_path, "wb") as f:
                for chunk in r.iter_content(chunk_size=8192):
                    if not chunk:
                        continue
                    f.write(chunk)
                    if progress_cb and total > 0:
                        downloaded += len(chunk)
                        pct = int(downloaded * 100 / total)
                        if pct != last_pct and (pct % 5 == 0 or pct in (0, 100)):
                            last_pct = pct
                            try:
                                progress_cb(pct)
                            except Exception:
                                # Même logique : jamais faire planter le download à cause de la gauge
                                pass
        if progress_cb:
            try:
                progress_cb(100)
            except Exception:
                pass




def get_torrent_config() -> Optional[Config]:
    """Construit la config torrent à partir des variables d'environnement."""
    if not TORRENT_SITE_BASE_URL or not TORRENT_USERNAME or not TORRENT_PASSWORD:
        logger.warning(
            "Config torrent incomplète (TORRENT_SITE_BASE_URL / TORRENT_USERNAME / TORRENT_PASSWORD)."
        )
        return None

    # On se base sur la même instance Flaresolverr que pour ZT
    flaresolverr_url = FLARESOLVERR_URL
    download_dir = DOWNLOAD_DIR_FILMS  # valeur par défaut, ajustée ensuite en fonction du mode

    return Config(
        site_base_url=TORRENT_SITE_BASE_URL,
        flaresolverr_url=flaresolverr_url,
        site_username=TORRENT_USERNAME,
        site_password=TORRENT_PASSWORD,
        default_category=TORRENT_CATEGORY_FILM,
        default_sub_category=TORRENT_SUBCATEGORY_FILM,
        download_dir=download_dir,
        alldebrid_api_key=ALLDEBRID_API_KEY or None,
        telegram_token=None,
    )


def torrent_search_sync(query: str, kind: str) -> List[TorrentResult]:
    """Recherche sur l'index torrent en fonction du type film/série."""
    cfg = get_torrent_config()
    if cfg is None:
        raise RuntimeError(
            "Config torrent incomplète. Vérifie TORRENT_SITE_BASE_URL / TORRENT_USERNAME / TORRENT_PASSWORD."
        )

    if kind == "film":
        category = TORRENT_CATEGORY_FILM
        sub_category = TORRENT_SUBCATEGORY_FILM
    else:
        category = TORRENT_CATEGORY_SERIE
        sub_category = TORRENT_SUBCATEGORY_SERIE

    fs = FlaresolverrClient(cfg.flaresolverr_url)
    site = SiteClient(cfg, fs)

    try:
        site.login()
        results = site.search(query, category=category, sub_category=sub_category)
        logger.info("Torrent search '%s' (%s) -> %d résultats", query, kind, len(results))
        return results
    finally:
        fs.destroy_session()


def torrent_download_sync(selected: TorrentResult, kind: str, progress_cb=None) -> Optional[str]:
    """Télécharge un résultat torrent via AllDebrid dans le bon dossier (films/séries).

    Retourne le dossier racine final (ou le chemin du .torrent si pas de clé AllDebrid).
    """
    cfg = get_torrent_config()
    if cfg is None:
        raise RuntimeError(
            "Config torrent incomplète. Vérifie TORRENT_SITE_BASE_URL / TORRENT_USERNAME / TORRENT_PASSWORD."
        )

    root_dir = DOWNLOAD_DIR_FILMS if kind == "film" else DOWNLOAD_DIR_SERIES
    cfg.download_dir = root_dir

    fs = FlaresolverrClient(cfg.flaresolverr_url)
    site = SiteClient(cfg, fs)

    detail_url = getattr(selected, "detail_url", None)
    if not detail_url:
        raise RuntimeError("Pas de detail_url pour ce résultat torrent.")

    try:
        site.login()
        download_url = site.get_download_url_from_detail(detail_url)
        if not download_url:
            raise RuntimeError("Impossible de récupérer le lien .torrent pour ce résultat.")
        logger.info("Lien .torrent récupéré : %s", download_url)

        tmp_dir = os.path.join(root_dir, "_torrents")
        os.makedirs(tmp_dir, exist_ok=True)

        filename = os.path.basename(urlparse(download_url).path) or "download.torrent"
        if not filename.lower().endswith(".torrent"):
            filename += ".torrent"
        torrent_path = os.path.join(tmp_dir, filename)

        fs.download_file(download_url, torrent_path)
        logger.info("Fichier .torrent téléchargé dans %s", torrent_path)
    finally:
        fs.destroy_session()

    if not ALLDEBRID_API_KEY:
        logger.warning(
            "ALLDEBRID_API_KEY manquante, seul le .torrent a été téléchargé : %s", torrent_path
        )
        return torrent_path

    ad = AllDebridClient(ALLDEBRID_API_KEY)
    magnet_id, ready = ad.upload_torrent_file(torrent_path)
    logger.info("Magnet AllDebrid créé (ID=%s, ready=%s), attente du statut Ready…", magnet_id, ready)
    magnet_info = ad.wait_until_ready(magnet_id)
    files = ad.get_files_links(magnet_id)

    if not files:
        logger.warning("Aucun fichier retourné par AllDebrid pour le magnet %s", magnet_id)
        return None

    real_name = magnet_info.get("filename") or os.path.splitext(os.path.basename(torrent_path))[0]
    safe_name = sanitize_filename(real_name)
    target_root = os.path.join(root_dir, safe_name)

    for f in files:
        # Nom relatif du fichier dans le torrent
        rel_name = f.get("name") or "file"

        # Je ne garde pas les .nfo, ça ne sert à rien et ça encombre juste le disque
        if rel_name.lower().endswith(".nfo"):
            logger.info("AllDebrid torrent : skip fichier .nfo -> %s", rel_name)
            continue

        page_link = f.get("link")
        if not page_link:
            continue

        # Le lien renvoyé par /magnet/files est une page AllDebrid (type https://alldebrid.com/f/XXXX)
        # Pour éviter de télécharger de l'HTML à la place de la vraie vidéo,
        # je passe systématiquement par notre helper alldebrid_unlock_sync,
        # qui utilise /link/redirector + /link/unlock si nécessaire.
        info = alldebrid_unlock_sync(page_link)
        if not info or not info.get("link"):
            logger.warning("AllDebrid torrent : impossible de débrider %s (%s)", rel_name, page_link)
            continue

        direct_url = info["link"]
        dest_path = os.path.join(target_root, rel_name)

        def file_progress(pct: int, name=rel_name):
            # Callback utilisé côté Telegram pour afficher "téléchargement de <name> - X%%"
            if progress_cb:
                progress_cb(name, pct)

        ad.download_direct(direct_url, dest_path, progress_cb=file_progress)

    logger.info("Téléchargement AllDebrid terminé dans %s", target_root)
    return target_root

# ======= UI HELPERS =======
def sanitize_filename(name: str) -> str:
    """Petit nettoyage de nom de fichier pour éviter les caractères foireux."""
    name = re.sub(r"[\\/:*?\"<>|]+", " ", name)
    name = re.sub(r"\s+", " ", name).strip()
    # Je limite un peu la longueur, histoire d'éviter des chemins trop longs
    return name[:150]


def main_menu_keyboard() -> InlineKeyboardMarkup:
    """
    Menu principal persistant :
    - Film / Série via ZT
    - Film / Série via Torrent
    - Annuler
    """
    return InlineKeyboardMarkup(
        [
            [
                InlineKeyboardButton("🎬 Film (ZT)", callback_data="mode_film_zt"),
                InlineKeyboardButton("📺 Série (ZT)", callback_data="mode_serie_zt"),
            ],
            [
                InlineKeyboardButton("🎬 Film (Torrent)", callback_data="mode_film_torrent"),
                InlineKeyboardButton("📺 Série (Torrent)", callback_data="mode_serie_torrent"),
            ],
            [
                InlineKeyboardButton("❌ Annuler", callback_data="cancel"),
            ],
        ]
    )



async def send_info(context: ContextTypes.DEFAULT_TYPE, chat_id: int, text: str):
    logger.info("CHAT %s | %s", chat_id, text.replace("\n", " "))
    await context.bot.send_message(chat_id=chat_id, text=text)


# ======= HTTP / FLARESOLVERR =======
def build_flareproxy_payload(url: str) -> dict:
    return {
        "cmd": "request.get",
        "url": url,
        "maxTimeout": 60000,
    }


def fetch_with_flaresolverr_sync(url: str) -> str:
    """
    Appel à Flaresolverr pour récupérer le HTML d'une page protégée.
    (Utilisé pour ZT + debug DL-Protect)
    """
    logger.info("Flaresolverr GET: %s", url)
    resp = session.post(FLARESOLVERR_URL, json=build_flareproxy_payload(url), timeout=60)
    resp.raise_for_status()
    data = resp.json()
    if data.get("status") != "ok":
        raise RuntimeError(f"Flaresolverr error: {data}")
    return data["solution"]["response"]


# ======= ALLDEBRID =======
def _alldebrid_headers() -> Dict[str, str]:
    if not ALLDEBRID_API_KEY:
        raise RuntimeError("ALLDEBRID_API_KEY manquant")
    return {
        "Authorization": f"Bearer {ALLDEBRID_API_KEY}",
        "User-Agent": ALLDEBRID_AGENT or "dhdflix",
    }


def alldebrid_unlock_sync(link: str) -> Optional[dict]:
    """
    Débride n'importe quel lien via AllDebrid.

    - Si c'est un lien de protecteur / redirector (ex: dl-protect.link),
      on passe d'abord par /link/redirector pour obtenir des liens encryptés
      redirect.alldebrid.com, puis on appelle /link/unlock dessus.
    - Si c'est un lien host direct (1fichier, rapidgator, ...),
      on appelle directement /link/unlock.
    """
    if not link:
        return None

    parsed = urlparse(link)
    host = (parsed.netloc or "").lower()

    headers = _alldebrid_headers()

    def _post(endpoint: str, data: Dict[str, str]) -> dict:
        url = f"https://api.alldebrid.com/v4/{endpoint}"
        r = session.post(url, headers=headers, data=data, timeout=60)
        r.raise_for_status()
        resp = r.json()
        return resp

    is_redirector = (
        "dl-protect" in host
        or "dlprotect" in host
        or "dl-protect1" in host
        or "dl-protect.net" in host
    )

    if is_redirector:
        logger.info("AllDebrid redirector flow pour: %s", link)
        try:
            resp = _post("link/redirector", {"link": link})
        except Exception as e:
            logger.warning("AllDebrid redirector error pour %s: %s", link, e)
            return None

        if resp.get("status") != "success":
            logger.warning("AllDebrid redirector API error: %s", resp)
            return None

        data = resp.get("data") or {}
        links = data.get("links") or []
        if not links:
            logger.warning("AllDebrid redirector: aucun lien retourné pour %s", link)
            return None

        encrypted_link = links[0]
        logger.info("AllDebrid redirector -> encrypted link: %s", encrypted_link)

        try:
            resp_unlock = _post("link/unlock", {"link": encrypted_link})
        except Exception as e:
            logger.warning("AllDebrid unlock (après redirector) error: %s", e)
            return None

        if resp_unlock.get("status") != "success":
            logger.warning("AllDebrid unlock error (après redirector): %s", resp_unlock)
            return None

        info = resp_unlock.get("data") or {}
        if not info.get("link"):
            logger.warning("AllDebrid unlock sans lien direct (après redirector): %s", resp_unlock)
            return None

        logger.info("AllDebrid final link (via redirector) pour %s -> %s", link, info.get("link"))
        return info

    # --- Cas host direct ---
    logger.info("AllDebrid unlock direct pour: %s", link)

    try:
        resp_unlock = _post("link/unlock", {"link": link})
    except Exception as e:
        logger.warning("AllDebrid unlock error pour %s: %s", link, e)
        return None

    if resp_unlock.get("status") != "success":
        logger.warning("AllDebrid unlock error: %s", resp_unlock)
        return None

    info = resp_unlock.get("data") or {}
    if not info.get("link"):
        logger.warning("AllDebrid unlock sans lien direct: %s", resp_unlock)
        return None

    logger.info("AllDebrid final link pour %s -> %s", link, info.get("link"))
    return info


def download_direct_sync(url: str, dest_dir: str, filename: str, progress_cb=None) -> str:
    """Téléchargement direct (ZT/hosts) avec callback de progression optionnel."""
    os.makedirs(dest_dir, exist_ok=True)
    path = os.path.join(dest_dir, filename)
    with session.get(url, stream=True, timeout=600) as r:
        r.raise_for_status()
        total = int(r.headers.get("Content-Length") or 0)
        downloaded = 0
        last_pct = -1
        with open(path, "wb") as f:
            for chunk in r.iter_content(chunk_size=1024 * 1024):
                if not chunk:
                    continue
                f.write(chunk)
                if progress_cb and total > 0:
                    downloaded += len(chunk)
                    pct = int(downloaded * 100 / total)
                    # J'évite de spammer Telegram : j'envoie quand ça bouge vraiment
                    if pct != last_pct and (pct % 5 == 0 or pct in (0, 100)):
                        last_pct = pct
                        try:
                            progress_cb(pct)
                        except Exception:
                            # La gauge ne doit jamais faire planter le download
                            pass
    # Au cas où on n'aurait jamais envoyé 100%
    if progress_cb:
        try:
            progress_cb(100)
        except Exception:
            pass
    return path


async def alldebrid_unlock(link: str) -> Optional[dict]:
    return await asyncio.to_thread(alldebrid_unlock_sync, link)


async def download_direct(url: str, dest_dir: str, filename: str, progress_cb=None) -> str:
    """Wrapper async pour download_direct_sync, avec callback de progression."""
    return await asyncio.to_thread(download_direct_sync, url, dest_dir, filename, progress_cb)


# ======= PARSING SEARCH =======
def parse_search_results(html: str, is_series: bool) -> List[SearchResult]:
    """
    Parse la page de recherche (films/séries) et retourne une liste de SearchResult.
    """
    soup = BeautifulSoup(html, "html.parser")
    results: List[SearchResult] = []

    for cover in soup.find_all("div", class_="cover_global"):
        info = cover.find("div", class_="cover_infos_title")
        if not info:
            continue
        a = info.find("a", href=True)
        if not a:
            continue

        href = a["href"]
        url = urljoin(INDEX_BASE_URL + "/", href)

        kind = None
        href_lower = href.lower()
        if "p=film" in href_lower:
            kind = "film"
        elif "p=serie" in href_lower:
            kind = "serie"

        if is_series and kind != "serie":
            continue
        if not is_series and kind != "film":
            continue

        title = a.get_text(strip=True)
        span = info.find("span", class_="detail_release")
        quality = span.get_text(strip=True) if span else ""

        results.append(
            SearchResult(
                title=title,
                quality=quality,
                url=url,
                is_series=is_series,
            )
        )

    results.sort(key=lambda r: (r.title.lower(), r.quality.lower()))
    return results


def _extract_pagination_urls(html: str, base_search_url: str) -> List[str]:
    """Analyse le HTML de la page 1 et extrait toutes les URLs de pagination
    correspondant à la même recherche (&search=...) avec un paramètre page.
    Retourne toujours au minimum [base_search_url]."""
    try:
        soup = BeautifulSoup(html, "html.parser")
    except Exception:
        return [base_search_url]

    urls: set[str] = {base_search_url}

    try:
        parsed = urlparse(base_search_url)
        base_search = parse_qs(parsed.query).get("search", [""])[0]
    except Exception:
        base_search = ""

    for a in soup.find_all("a", href=True):
        href = a["href"]
        if "page=" not in href or "search=" not in href:
            continue

        parsed = urlparse(href)
        qs = parse_qs(parsed.query)
        if base_search:
            search_val = qs.get("search", [""])[0]
            if search_val != base_search:
                continue

        full = href
        if not full.startswith("http"):
            full = urljoin(INDEX_BASE_URL + "/", full.lstrip("/"))
        urls.add(full)

    def page_key(u: str) -> int:
        try:
            p = urlparse(u)
            return int(parse_qs(p.query).get("page", ["1"])[0])
        except Exception:
            return 1

    return sorted(urls, key=page_key)


# ======= PARSING EPISODES / FILMS =======
def extract_episode_links(html: str) -> Dict[str, Dict[str, str]]:
    """
    Pour une page de SÉRIE : renvoie un dict:
    {
      "1": {"Uploady": "...", "DailyUploads": "...", "1fichier": "...", ...},
      "2": {...},
      ...
    }

    On NE prend que le bloc "Liens De Téléchargement" (pas le streaming).
    Et on associe chaque lien à l'hébergeur en fonction du dernier <div style="font-weight:bold">
    rencontré avant le lien.
    """
    soup = BeautifulSoup(html, "html.parser")
    episodes_by_host: Dict[str, Dict[str, str]] = {}

    # 1) On localise le H2 "Liens De Téléchargement" (section DDL)
    h2 = soup.find("h2", string=lambda t: t and "Liens De Téléchargement" in t)
    if not h2:
        return {}

    # 2) On prend le premier div id="news-id-23557" APRÈS ce h2
    container = h2.find_next("div", id="news-id-23557")
    if not container:
        return {}

    current_host: Optional[str] = None

    # 3) On parcourt les descendants dans l'ordre
    for tag in container.descendants:
        if not hasattr(tag, "name"):
            continue

        # Nouveau host ?
        if tag.name == "div":
            style = (tag.get("style") or "").lower()
            if "font-weight:bold" in style:
                # ex: "Uploady", "DailyUploads", "Rapidgator", "1fichier", "Vidoza", ...
                host_name = tag.get_text(strip=True)
                host_name = re.sub(r"\s+", " ", host_name)
                current_host = host_name
                continue

        # Lien d'épisode ?
        if tag.name == "a" and current_host:
            href = tag.get("href", "")
            text = tag.get_text(strip=True)

            if "dl-protect" not in href.lower():
                continue

            m = re.search(r"episode\s*(\d+)", text, re.I)
            if not m:
                continue

            ep_num = m.group(1)
            episodes_by_host.setdefault(ep_num, {})
            episodes_by_host[ep_num][current_host] = href

    return episodes_by_host


def extract_film_links(html: str) -> Dict[str, str]:
    """
    Pour une page de FILM :
    renvoie {host_name: url_dl_protect}
    en se basant également sur le bloc "Liens De Téléchargement".
    """
    soup = BeautifulSoup(html, "html.parser")
    mapping: Dict[str, str] = {}

    h2 = soup.find("h2", string=lambda t: t and "Liens De Téléchargement" in t)
    if not h2:
        return {}

    container = h2.find_next("div", id="news-id-23557")
    if not container:
        return {}

    current_host: Optional[str] = None

    for tag in container.descendants:
        if not hasattr(tag, "name"):
            continue

        if tag.name == "div":
            style = (tag.get("style") or "").lower()
            if "font-weight:bold" in style:
                host_name = tag.get_text(strip=True)
                host_name = re.sub(r"\s+", " ", host_name)
                current_host = host_name
                continue

        if tag.name == "a" and current_host:
            href = tag.get("href", "")
            text = tag.get_text(strip=True)

            if "dl-protect" not in href.lower():
                continue

            # On ne veut que les liens "Télécharger" (pas "Episode X")
            if "télécharger" not in text.lower() and "telecharger" not in text.lower():
                # pour certains films, il n'y a que "Partie 1" etc, donc
                # si tu veux élargir, tu peux enlever ce check
                continue

            mapping[current_host] = href

    return mapping


# ======= DEBUG DL-PROTECT =======
async def send_protect_html_debug(
    context: ContextTypes.DEFAULT_TYPE,
    chat_id: int,
    label: str,
    protect_url: str,
) -> None:
    """
    Télécharge le HTML de la page dl-protect et l'envoie comme fichier
    dans Telegram pour debug.
    """
    try:
        html = await asyncio.to_thread(fetch_with_flaresolverr_sync, protect_url)
    except Exception as e:
        logger.warning("send_protect_html_debug: échec Flaresolverr pour %s: %s", protect_url, e)
        return

    try:
        encoded = html.encode("utf-8", errors="ignore")
        if len(encoded) > 500_000:
            encoded = encoded[:500_000]

        bio = io.BytesIO(encoded)
        bio.name = f"dlprotect_debug_{label}.html"

        await context.bot.send_document(
            chat_id=chat_id,
            document=bio,
            filename=bio.name,
            caption=f"HTML dl-protect pour {label} (debug)",
        )
    except Exception as e:
        logger.warning("send_protect_html_debug: impossible d'envoyer le HTML pour %s: %s", label, e)


# ======= TELEGRAM HANDLERS =======
async def start(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """
    /start ou retour menu : on réinitialise le flag d'abort
    et on affiche le menu principal.
    """
    chat_id = update.effective_chat.id
    context.user_data["abort"] = False
    context.user_data.pop("mode", None)
    context.user_data.pop("mode_type", None)
    context.user_data.pop("mode_source", None)
    context.user_data.pop("results", None)
    context.user_data.pop("query", None)

    if update.message:
        await update.message.reply_text(
            "Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )
    else:
        await context.bot.send_message(
            chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )

    return MODE



async def mode_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    query = update.callback_query
    await query.answer()
    data = query.data

    if not data.startswith("mode_"):
        await query.edit_message_text("Mode inconnu.")
        return MODE

    parts = data.split("_")
    if len(parts) != 3:
        await query.edit_message_text("Mode inconnu.")
        return MODE

    _, kind, source = parts
    if kind not in ("film", "serie") or source not in ("zt", "torrent"):
        await query.edit_message_text("Mode inconnu.")
        return MODE

    # Pour compatibilité avec l'ancien code ZT
    if source == "zt":
        context.user_data["mode"] = kind

    context.user_data["mode_type"] = kind
    context.user_data["mode_source"] = source

    if kind == "film":
        if source == "zt":
            text = "Envoie-moi le nom du film à chercher sur ZT :"
        else:
            text = "Envoie-moi le nom du film à chercher en torrent :"
    else:
        if source == "zt":
            text = "Envoie-moi le nom de la série à chercher sur ZT :"
        else:
            text = "Envoie-moi le nom de la série à chercher en torrent :"

    await query.edit_message_text(text)
    return SEARCH



async def cancel_inline(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """
    Callback sur le bouton "❌ Annuler".
    On se contente de poser le flag 'abort' à True.
    Les fonctions de download checkent ce flag et s'arrêtent.
    """
    query = update.callback_query
    await query.answer()
    chat_id = query.message.chat_id

    context.user_data["abort"] = True
    await query.edit_message_text("❌ Annulation demandée, j'arrête la requête en cours...")

    # On laisse le handler de téléchargement détecter abort et
    # le choose_result_callback renverra ensuite le menu principal.
    return MODE


async def cancel_command(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """
    Commande /cancel (fallback ConversationHandler).
    Même logique que cancel_inline : on pose juste le flag.
    """
    chat_id = update.effective_chat.id
    context.user_data["abort"] = True
    await update.message.reply_text("❌ Annulation demandée, j'arrête la requête en cours...")

    return MODE


async def zt_search_query(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    mode = context.user_data.get("mode")
    chat_id = update.message.chat_id
    if mode not in ("film", "serie"):
        await send_info(context, chat_id, "Je ne sais pas encore si tu veux un film ou une série. /start")
        return MODE

    query_text = update.message.text.strip()
    context.user_data["query"] = query_text
    context.user_data["abort"] = False  # nouvelle recherche, on reset l'abort

    await send_info(context, chat_id, f"Recherche de {mode} pour : {query_text}")

    if mode == "film":
        p_value = "films"
    else:
        p_value = "series"

    search_url = f"{INDEX_BASE_URL}/?p={p_value}&search={quote(query_text)}"
    is_series = (mode == "serie")

    async def do_search() -> List[SearchResult]:
        html = await asyncio.to_thread(fetch_with_flaresolverr_sync, search_url)
        if not html:
            logger.warning("HTML vide pour la recherche %s", search_url)
            return []

        results = parse_search_results(html, is_series=is_series)

        if not results:
            debug_path = "/app/debug_last_search.html"
            try:
                with open(debug_path, "w", encoding="utf-8") as f:
                    f.write(html)
                logger.warning("Aucun résultat page 1 -> HTML sauvegardé dans %s", debug_path)
            except Exception as e:
                logger.warning("Impossible de sauvegarder le HTML de debug: %s", e)

        page_urls = _extract_pagination_urls(html, search_url)

        for page_url in page_urls[1:]:
            if context.user_data.get("abort"):
                logger.info("Recherche interrompue (abort=True)")
                return results

            logger.info("Chargement page de résultats supplémentaire : %s", page_url)
            html_page = await asyncio.to_thread(fetch_with_flaresolverr_sync, page_url)
            if not html_page:
                continue
            add_results = parse_search_results(html_page, is_series=is_series)
            if not add_results:
                continue
            results.extend(add_results)

        return results

    try:
        results = await do_search()
    except Exception as e:
        await send_info(context, chat_id, f"Erreur pendant la recherche: {e}")
        return MODE

    if context.user_data.get("abort"):
        await send_info(context, chat_id, "Recherche annulée.")
        # On renverra le menu via /start ou explicitement si tu veux
        await context.bot.send_message(
            chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )
        return MODE

    if not results:
        await send_info(context, chat_id, "Aucun résultat trouvé.")
        await context.bot.send_message(
            chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )
        return MODE

    context.user_data["results"] = results

    keyboard = []
    for idx, r in enumerate(results[:30], start=1):
        label = f"{r.title} {r.quality}".strip()
        keyboard.append(
            [InlineKeyboardButton(label[:60], callback_data=f"choose_{idx-1}")]
        )

    keyboard.append([InlineKeyboardButton("❌ Annuler", callback_data="cancel")])

    await update.message.reply_text(
        "Résultats trouvés, choisis :", reply_markup=InlineKeyboardMarkup(keyboard)
    )
    return CHOOSE_RESULT


async def torrent_search_query(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """Recherche via l'index torrent (mode film/série déjà choisi)."""
    chat_id = update.message.chat_id
    kind = context.user_data.get("mode_type")

    if kind not in ("film", "serie"):
        await send_info(
            context,
            chat_id,
            "Je ne sais pas encore si tu veux un film ou une série (torrent). /start",
        )
        return MODE

    query_text = update.message.text.strip()
    context.user_data["query"] = query_text
    context.user_data["abort"] = False

    await send_info(context, chat_id, f"Recherche torrent de {kind} pour : {query_text}")

    try:
        results = await asyncio.to_thread(torrent_search_sync, query_text, kind)
    except Exception as e:
        logger.exception("Erreur pendant la recherche torrent", exc_info=True)
        await send_info(context, chat_id, f"Erreur pendant la recherche torrent : {e}")
        await context.bot.send_message(
            chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )
        return MODE

    if not results:
        await send_info(context, chat_id, "Aucun résultat torrent trouvé.")
        await context.bot.send_message(
            chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )
        return MODE

    context.user_data["results"] = results

    keyboard = []
    for idx, r in enumerate(results[:30], start=1):
        size = r.size or "?"
        label = f"[{size}] {r.title}"
        keyboard.append(
            [InlineKeyboardButton(label[:60], callback_data=f"choose_{idx-1}")]
        )

    keyboard.append([InlineKeyboardButton("❌ Annuler", callback_data="cancel")])

    await update.message.reply_text(
        "Résultats trouvés, choisis :", reply_markup=InlineKeyboardMarkup(keyboard)
    )
    return CHOOSE_RESULT


async def search_query(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """Point d'entrée commun : route vers ZT ou torrent suivant mode_source."""
    source = context.user_data.get("mode_source")
    if source == "torrent":
        return await torrent_search_query(update, context)
    else:
        # Par défaut (None ou "zt"), on reste sur la logique ZT historique
        return await zt_search_query(update, context)



async def choose_result_callback(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    query = update.callback_query
    await query.answer()
    chat_id = query.message.chat_id
    data = query.data

    if not data.startswith("choose_"):
        await query.edit_message_text("Choix invalide.")
        await context.bot.send_message(
            chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )
        return MODE

    idx = int(data.split("_", 1)[1])
    results = context.user_data.get("results", [])
    if not isinstance(results, list) or not (0 <= idx < len(results)):
        await query.edit_message_text("Choix invalide (index hors limite).")
        await context.bot.send_message(
            chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
        )
        return MODE

    selected = results[idx]
    source = context.user_data.get("mode_source", "zt")

    # Texte différent selon la source
    if source == "torrent":
        desc = getattr(selected, "title", str(selected))
    else:
        title = getattr(selected, "title", "")
        quality = getattr(selected, "quality", "")
        desc = f"{title} {quality}".strip()

    await query.edit_message_text(f"Tu as choisi : {desc}")

    if source == "torrent":
        await handle_torrent_download(chat_id, selected, context)
    else:
        # Logique ZT historique
        if getattr(selected, "is_series", False):
            await handle_series_download(chat_id, selected, context)
        else:
            await handle_film_download(chat_id, selected, context)

    # À la fin de TOUT (succès, échec ou abort), on renvoie le menu principal
    await context.bot.send_message(
        chat_id=chat_id, text="Que veux-tu chercher ?", reply_markup=main_menu_keyboard()
    )
    return MODE



async def handle_torrent_download(
    chat_id: int,
    selected: TorrentResult,
    context: ContextTypes.DEFAULT_TYPE,
):
    """Téléchargement d'un résultat torrent via AllDebrid dans le bon dossier (films/séries)."""
    # Même logique d'abort que pour ZT
    if context.user_data.get("abort"):
        await send_info(context, chat_id, "Téléchargement annulé.")
        return

    kind = context.user_data.get("mode_type", "film")
    await send_info(context, chat_id, "Analyse du lien torrent et envoi vers AllDebrid…")

    # Message principal de gauge pour tout le torrent
    progress_msg = await context.bot.send_message(
        chat_id=chat_id,
        text="Torrent : préparation du téléchargement…",
    )
    loop = asyncio.get_running_loop()

    def torrent_progress(name: str, pct: int) -> None:
        """Gauge globale du torrent : on affiche le fichier courant + le pourcentage."""
        try:
            loop.call_soon_threadsafe(
                lambda: asyncio.create_task(
                    safe_edit_message(
                        progress_msg,
                        f"Torrent : téléchargement de {name} - {pct}%",
                    )
                )
            )
        except Exception:
            pass

    try:
        target_root = await asyncio.to_thread(torrent_download_sync, selected, kind, torrent_progress)
    except Exception as e:
        logger.exception("Erreur pendant le téléchargement torrent", exc_info=True)
        try:
            await safe_edit_message(
                progress_msg,
                f"Torrent : erreur pendant le téléchargement ({e})",
            )
        except Exception:
            pass
        await send_info(context, chat_id, f"Erreur pendant le téléchargement via torrent : {e}")
        return

    if target_root:
        try:
            await safe_edit_message(
                progress_msg,
                "Torrent : téléchargement terminé - 100%",
            )
        except Exception:
            pass
        await send_info(context, chat_id, f"✅ Téléchargement terminé dans : {target_root}")
    else:
        try:
            await safe_edit_message(
                progress_msg,
                "Torrent : aucun fichier téléchargé (réponse AllDebrid vide ?)",
            )
        except Exception:
            pass
        await send_info(context, chat_id, "Aucun fichier téléchargé (réponse AllDebrid vide ?).")



async def safe_edit_message(message, text: str) -> None:
    """Édite un message Telegram en absorbant les erreurs 'Message is not modified'."""
    try:
        await message.edit_text(text)
    except BadRequest as e:
        # Cas très courant avec les gauges: même texte -> on ignore
        if "Message is not modified" in str(e):
            return
        logger.warning("Erreur lors de l'édition du message: %s", e)
    except Exception as e:
        logger.warning("Erreur inattendue lors de l'édition du message: %s", e)


async def handle_film_download(
    chat_id: int,
    selected: SearchResult,
    context: ContextTypes.DEFAULT_TYPE,
):
    if context.user_data.get("abort"):
        await send_info(context, chat_id, "Téléchargement annulé.")
        return

    await send_info(context, chat_id, "Analyse des liens du film…")
    html = await asyncio.to_thread(fetch_with_flaresolverr_sync, selected.url)
    host_links = extract_film_links(html)

    if not host_links:
        await send_info(context, chat_id, "Aucun lien de téléchargement trouvé pour ce film.")
        return

    hosts_order = ["1fichier", "Uploady", "DailyUploads", "Turbobit", "Rapidgator", "Nitroflare"]
    hosts = [h for h in hosts_order if h in host_links] + [
        h for h in host_links.keys() if h not in hosts_order
    ]

    base_dir = os.path.join(
        DOWNLOAD_DIR_FILMS,
        sanitize_filename(f"{selected.title} {selected.quality}".strip()),
    )

    for host in hosts:
        if context.user_data.get("abort"):
            await send_info(context, chat_id, "Téléchargement annulé.")
            return

        link = host_links[host]
        await send_info(context, chat_id, f"Film - {host} : envoi du lien à AllDebrid…")
        logger.info("Film - %s : AllDebrid unlock sur %s", host, link)

        data = await alldebrid_unlock(link)
        if not data:
            await send_info(
                context,
                chat_id,
                f"Film - {host} : échec du débridage, essai hébergeur suivant.",
            )
            if "dl-protect" in link:
                await send_protect_html_debug(
                    context,
                    chat_id,
                    label=f"film_{host}_alldebrid_error",
                    protect_url=link,
                )
            continue

        direct = data.get("link")
        filename = data.get("filename") or os.path.basename(direct.split("?")[0]) or "film.mkv"
        filename = sanitize_filename(filename)

        # Message qui sert aussi de gauge de progression pour ce film
        progress_msg = await context.bot.send_message(
            chat_id=chat_id,
            text=f"Film - {host} : téléchargement de {filename} - 0%",
        )
        loop = asyncio.get_running_loop()

        def progress_cb(pct: int) -> None:
            """Callback appelé depuis le thread de download pour mettre à jour la gauge du film."""
            try:
                loop.call_soon_threadsafe(
                    lambda: asyncio.create_task(
                        safe_edit_message(
                            progress_msg,
                            f"Film - {host} : téléchargement de {filename} - {pct}%",
                        )
                    )
                )
            except Exception:
                # La gauge ne doit jamais faire planter le download
                pass

        try:
            path = await download_direct(direct, base_dir, filename, progress_cb=progress_cb)
        except Exception as e:
            try:
                await safe_edit_message(
                    progress_msg,
                    f"Film - {host} : téléchargement de {filename} - erreur ({e}), essai hébergeur suivant.",
                )
            except Exception:
                pass
            await send_info(
                context,
                chat_id,
                f"Film - {host} : échec téléchargement ({e}), essai hébergeur suivant.",
            )
            continue

        try:
            await safe_edit_message(
                progress_msg,
                f"Film - {host} : téléchargement de {filename} - 100%",
            )
        except Exception:
            pass
        await send_info(context, chat_id, f"✅ Film téléchargé : {path}")
        return

    await send_info(context, chat_id, "❌ Impossible de télécharger le film depuis les hébergeurs disponibles.")


async def handle_series_download(
    chat_id: int,
    selected: SearchResult,
    context: ContextTypes.DEFAULT_TYPE,
):
    if context.user_data.get("abort"):
        await send_info(context, chat_id, "Téléchargement annulé.")
        return

    await send_info(context, chat_id, "Analyse des liens de la série…")
    html = await asyncio.to_thread(fetch_with_flaresolverr_sync, selected.url)
    episodes_by_host = extract_episode_links(html)

    if not episodes_by_host:
        await send_info(context, chat_id, "Impossible de trouver les épisodes pour cette série.")
        return

    ep_numbers = sorted(episodes_by_host.keys(), key=lambda x: int(x))

    hosts_order = ["Uploady", "DailyUploads", "Rapidgator", "Nitroflare", "Turbobit", "1fichier", "Vidoza"]

    base_dir = os.path.join(
        DOWNLOAD_DIR_SERIES,
        sanitize_filename(f"{selected.title} {selected.quality}".strip()),
    )

    for ep in ep_numbers:
        if context.user_data.get("abort"):
            await send_info(context, chat_id, "Téléchargement annulé.")
            return

        host_links = episodes_by_host[ep]
        success = False

        for host in hosts_order:
            if context.user_data.get("abort"):
                await send_info(context, chat_id, "Téléchargement annulé.")
                return

            if host not in host_links:
                continue

            link = host_links[host]
            label = f"Episode {ep} - {host}"

            await send_info(context, chat_id, f"{label} : envoi du lien à AllDebrid…")
            logger.info("%s : AllDebrid unlock sur %s", label, link)

            data = await alldebrid_unlock(link)
            if not data:
                await send_info(
                    context,
                    chat_id,
                    f"{label} : échec débridage, essai hébergeur suivant.",
                )
                if "dl-protect" in link:
                    await send_protect_html_debug(
                        context,
                        chat_id,
                        label=f"serie_{host}_ep{ep}_alldebrid_error",
                        protect_url=link,
                    )
                continue

            direct = data.get("link")
            filename = (
                data.get("filename")
                or os.path.basename(direct.split("?")[0])
                or f"S01E{ep}.mkv"
            )
            filename = sanitize_filename(filename)

            # Message qui servira aussi de gauge pour cet épisode
            progress_msg = await context.bot.send_message(
                chat_id=chat_id,
                text=f"{label} : téléchargement de {filename} - 0%",
            )
            loop = asyncio.get_running_loop()

            def progress_cb(pct: int) -> None:
                """Gauge de progression de l'épisode, appelée depuis le thread de download."""
                try:
                    loop.call_soon_threadsafe(
                        lambda: asyncio.create_task(
                            safe_edit_message(
                                progress_msg,
                                f"{label} : téléchargement de {filename} - {pct}%",
                            )
                        )
                    )
                except Exception:
                    pass

            try:
                path = await download_direct(direct, base_dir, filename, progress_cb=progress_cb)
            except Exception as e:
                try:
                    await safe_edit_message(
                        progress_msg,
                        f"{label} : téléchargement de {filename} - erreur ({e}), essai hébergeur suivant.",
                    )
                except Exception:
                    pass
                await send_info(
                    context,
                    chat_id,
                    f"{label} : échec téléchargement ({e}), essai hébergeur suivant.",
                )
                continue

            try:
                await safe_edit_message(
                    progress_msg,
                    f"{label} : téléchargement de {filename} - 100%",
                )
            except Exception:
                pass
            await send_info(context, chat_id, f"✅ {label} téléchargé : {path}")
            success = True
            break  # dès qu'un host OK pour cet épisode -> épisode suivant

        if not success:
            await send_info(
                context,
                chat_id,
                f"❌ Tous les hébergeurs ont échoué pour l'épisode {ep}.",
            )


async def cancel(update: Update, context: ContextTypes.DEFAULT_TYPE) -> int:
    """Fallback ConversationHandler -> /cancel, redirigé vers cancel_command."""
    return await cancel_command(update, context)


def main():
    if not TELEGRAM_TOKEN:
        raise SystemExit("TELEGRAM_TOKEN manquant")

    app = ApplicationBuilder().token(TELEGRAM_TOKEN).build()

    conv = ConversationHandler(
        entry_points=[CommandHandler("start", start)],
        states={
            MODE: [
                CallbackQueryHandler(mode_callback, pattern=r"^mode_"),
                CallbackQueryHandler(cancel_inline, pattern=r"^cancel$"),
                CommandHandler("start", start),
            ],
            SEARCH: [
                MessageHandler(filters.TEXT & ~filters.COMMAND, search_query),
                CommandHandler("start", start),
            ],
            CHOOSE_RESULT: [
                CallbackQueryHandler(choose_result_callback, pattern=r"^choose_"),
                CallbackQueryHandler(cancel_inline, pattern=r"^cancel$"),
                CommandHandler("start", start),
            ],
        },
        fallbacks=[CommandHandler("cancel", cancel)],
        allow_reentry=True,
    )

    app.add_handler(conv)

    logger.info("Bot démarré.")
    app.run_polling()


if __name__ == "__main__":
    main()