#!/usr/bin/env bash
set -euo pipefail

STATE_DIR="${HOME}/.cache/starship-net"
mkdir -p "$STATE_DIR"

# Une clé différente par terminal Starship
SESSION="${STARSHIP_SESSION_KEY:-$PPID}"
STATE_FILE="$STATE_DIR/$SESSION"

# Trouver l’interface réseau principale (celle de la route par défaut)
DEV="$(ip route get 1.1.1.1 2>/dev/null | awk '/dev/ {for (i=1;i<=NF;i++) if ($i=="dev") {print $(i+1); exit}}')"
if [ -z "${DEV}" ]; then
  DEV="$(ip route show default 2>/dev/null | awk '/dev/ {for (i=1;i<=NF;i++) if ($i=="dev") {print $(i+1); exit}}' | head -n1)"
fi

# Si on ne trouve rien, on renvoie N/A
if [ -z "${DEV}" ] || [ ! -r "/sys/class/net/${DEV}/statistics/rx_bytes" ]; then
  echo "N/A"
  exit 0
fi

rx_now="$(cat "/sys/class/net/${DEV}/statistics/rx_bytes")"
tx_now="$(cat "/sys/class/net/${DEV}/statistics/tx_bytes")"

# Première exécution de la session → on initialise la base et renvoie 0
if [ ! -f "${STATE_FILE}" ]; then
  echo "${rx_now} ${tx_now}" > "${STATE_FILE}"
  echo "0.00o↓ 0.00o↑"
  exit 0
fi

read rx_base tx_base < "${STATE_FILE}"

# Diff depuis le début de la session
drx=$((rx_now - rx_base))
dtx=$((tx_now - tx_base))

# Protection sous-flow si jamais (reset compteur)
if [ "${drx}" -lt 0 ]; then drx=0; fi
if [ "${dtx}" -lt 0 ]; then dtx=0; fi

# Fonction pour convertir en o / Ko / Mo / Go
human() {
  local b="$1"
  awk -v b="${b}" 'BEGIN {
    unit = "o"
    val  = b
    if (b >= 1024) {
      unit = "Ko"
      val  = b / 1024
    }
    if (b >= 1024*1024) {
      unit = "Mo"
      val  = b / (1024*1024)
    }
    if (b >= 1024*1024*1024) {
      unit = "Go"
      val  = b / (1024*1024*1024)
    }
    printf "%.2f%s", val, unit
  }'
}

down="$(human "${drx}")"
up="$(human "${dtx}")"

echo "${down}↓ ${up}↑"
